#!/usr/bin/env python
# -*- coding: utf-8 -*-
import logging
import os
import json
import pysolr
import redis
import pika
import re
import time
from .solr import SolrCoreManager
from .smkapi import SMKApi
from .consumer import DelegatingConsumer
from datetime import datetime, date
__version__ = "0.1.7"

LOG_FORMAT = ('%(levelname) -10s %(asctime)s %(name) -30s %(funcName) '
              '-35s %(lineno) -5d: %(message)s')
LOGGER = logging.getLogger(__name__)


class ReconnectingDelegatingConsumer(object):
    """This is an example consumer that will reconnect if the nested
    ExampleConsumer indicates that a reconnect is necessary.

    """
    def __init__(self, amqp_url, queue, routing_key,
                 exchange=None, exchange_type=None, delegate=None):

        self._amqp_url = amqp_url
        self._queue = queue
        self._routing_key = routing_key
        self._exchange = exchange
        self._exchange_type = exchange_type
        self._delegate = delegate
        self._consumer = DelegatingConsumer(
            self._amqp_url, self._queue, self._routing_key,
            self._exchange, self._exchange_type, self._delegate)


class ExampleApp(object):
    """ Example app that handles `artwork.[created|updated|deleted]` messages
        Implement your own artwork enricher by subclassing and
        overriding following methods:
        - `fetch`
        - `enrich`
        - `persist`
        - `get_cache_key`

        To handle other classes of messages override `get_handler`
        and the `artwork_` methods.
    """

    def __init__(self, appid, rabbitmq, queue, routing_key, exchange,
                 solr_address, redis_host=None, redis_port=None):
        self._reconnect_delay = 0
        self.api = SMKApi()
        self.appid = appid
        self.exchange = exchange
        self.rabbitmq = rabbitmq
        self.queue = queue
        self.routing_key = f'{routing_key} {routing_key}.{appid}'
        self.consumer = DelegatingConsumer(
            rabbitmq, queue, self.routing_key, exchange, delegate=self
        )
        self.scm = SolrCoreManager(solr_address, appid + '_core')
        if redis_host and redis_port:
            self.cache = redis.Redis(redis_host, redis_port)
        else:
            self.cache = None

    @property
    def solr(self):
        return self.scm.get_connection()

    def run(self):
        while True:
            try:
                self.consumer.run()
            except KeyboardInterrupt:
                self.consumer.stop()
                break
            self._maybe_reconnect()

    def _maybe_reconnect(self):
        if self.consumer.should_reconnect:
            self.consumer.stop()
            reconnect_delay = self._get_reconnect_delay()
            LOGGER.info('Reconnecting after %d seconds', reconnect_delay)
            time.sleep(reconnect_delay)
            self.consumer = DelegatingConsumer(
                self.rabbitmq, self.queue, self.routing_key,
                self.exchange, delegate=self
            )

    def _get_reconnect_delay(self):
        if self.consumer.was_consuming:
            self._reconnect_delay = 0
        else:
            self._reconnect_delay += 1
        if self._reconnect_delay > 30:
            self._reconnect_delay = 30
        return self._reconnect_delay

    def artwork_created(self, message):
        return self.artwork_updated(message)

    def artwork_updated(self, message):
        id = self.get_object_number(message)
        obj = self.fetch(id)
        if obj:
            data = self.cached_enrich(id, obj)
            self.persist(id, data)
        else:
            LOGGER.info(
                'No object found for id: %s', id)

    def artwork_deleted(self, message):
        id = self.get_object_number(message)
        LOGGER.info('Deleting Doc %s from Solr.', id)
        self.solr.delete(id, commit=True)

    def get_object_number(self, message):
        object_number = message.get('object_number')
        if not object_number:
            LOGGER.info(
                'Ignoring message without object_number! %s', message)
        return object_number

    def fetch(self, id):
        """ return an item from  """
        res = self.api.art(id)
        try:
            return res.get('items')[0]
        except (TypeError, IndexError):
            return None

    def cached_enrich(self, id, item):
        # TODO: move caching to decorator
        if not self.cache:
            return self.enrich(id, item)
        cache_key = self.get_cache_key(id, item)
        cached = self.cache.get(cache_key)
        if cached is not None:
            LOGGER.info('Using cached data for item "%s" with key "%s"',
                        id, cache_key)
            return json.loads(cached)

        data = self.enrich(id, item)

        self.cache.set(cache_key, json.dumps(data))
        return data

    def enrich(self, id, item):
        """ """
        if not item:
            return {}

        def get_title_str(lang):
            return ', '.join(
                [t.get('title', t.get('translation'))
                 for t in item.get('titles', [])
                 if t.get('language').startswith(lang)]
            )

        data = {
            'simple_da_s': get_title_str('da'),
            'simple_en_s': get_title_str('en'),
            'simple_neutral_s': get_title_str('neutral'),
            'tags_da_s': '',
            'tags_en_s': '',
            'tags_neutral_s': '',
        }
        return data

    def persist(self, id, data):
        LOGGER.info(f'Updating Doc %s in Solr.', id)
        data['id'] = id
        self.solr.add([data], commit=True)

        tags = {k: v for k, v in data.items() if k.startswith('tags_')}
        self.consumer.publish_message(json.dumps(
            {
                "object_number": id,
                "appId": self.appid,
                "tags": tags
            }
        ), 'tags.updated')

    def get_cache_key(self, id, item):
        return f'{self.appid}-{id}-{date.isoformat(datetime.now())}'

    def get_handler(self, routing_key):
        return {
            'artwork.updated': self.artwork_updated,
            'artwork.created': self.artwork_created,
            'artwork.deleted': self.artwork_deleted
        }.get(re.sub('\.%s$' % self.appid, '', routing_key))


def main():
    loglevel = getattr(logging, os.environ.get('LOGLEVEL', 'INFO'))
    logging.basicConfig(level=loglevel, format=LOG_FORMAT)

    rabbitmq = os.environ.get(
        'RABBITMQ', 'amqp://roger:jessica@rabbitmq:5672/%2F')
    appid = os.environ.get('appID', 'simplecomponent')
    solr_address = os.environ.get('SOLR_ADDRESS', 'http://solr:8983/solr')
    exchange = os.environ.get('EXCHANGE', 'smk')
    redis_host = os.environ.get('REDIS_HOST', 'redis')
    redis_port = os.environ.get('REDIS_PORT', '6379')
    queue = appid
    routing_key = "artwork.*"

    app = ExampleApp(
        appid, rabbitmq, queue, routing_key, exchange,
        solr_address, redis_host, redis_port
    )
    app.run()

if __name__ == '__main__':
    main()
